/** @file   weapon.h
 * @brief   Declaration of Weapon - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_WEAPON_H
#define H_WWW_WEAPON_H

#include "vec2d.h"
#include <vector>


namespace WeWantWar {



/** @class  Weapon
 * @brief   Represents and manages the weapons.
 * @author  Tomi Lamminsaari
 *
 * The weapons the GameObject carries are instances of this class. The
 * weapon specs cannot be changed during the gameplay but they are
 * initialized during the game startup process.
 *
 * To initialize the weapons you must call the static <code> init() </code> -
 * method before you are allowed to create any instances of the Weapons.
 *
 * This same class represents all the weapons the game have. They are
 * differentiated from each other by id-codes.
 */
class Weapon
{
public:

  /** Structure for technical details of the weapons. */
  struct Specs {
    /** Type of this weapon */
    int id;
    /** Reload delay. Grenades use this field to store the detonating delay. */
    int reload;
    /** How much damage the bullets cause. */
    int damage;
    /** Velocity of the bullets. */
    float speed;
    /** Shooting range in pixels. */
    float range;
    /** The size of the bullet. */
    float radius;
    /** The default deviation of the bullet torrent. The Grenade and tankbullet
     * uses this field store the number of fragments it emits as it explodes.
     */
    int deviation;
    /** Should we make a sound when hitting the tilemap */
    int hitSound;
    /** Should the bullets of this weapon use the bullet trails visual effect */
    int useTrails;
    /** Does the bullets of this weapon bounce from the walls. */
    int bounce;
  };
  
  
  /** A datatype for weapon id-codes. */
  typedef int WeaponID;
  
  /** ID of the crowbar-weapon */
  static const int W_CROWBAR = 0;
  /** ID of the rifle */
  static const int W_RIFLE = 1;
  /** ID of the shotgun */
  static const int W_SHOTGUN = 2;
  /** ID of the flamethrower */
  static const int W_FLAMETHROWER = 3;
  /** ID of the alien fireball. One of the weapons the aliens use. */
  static const int W_ALIENFIREBALL = 4;
  /** ID of the minigun */
  static const int W_MINIGUN = 5;
  /** ID of the dual uzis */
  static const int W_UZI = 6;
  /** ID of no-weapon */
  static const int W_NOWEAPON = 7;
  /** ID of Grenade */
  static const int W_GRENADE = 8;
  /** ID of Tank Bullet */
  static const int W_TANKBULLET = 9;
  /** ID of sentrygun fireball */
  static const int W_SENTRYFIREBALL = 10;
  /** ID of sniper-gun */
  static const int W_SNIPERRIFLE = 11;
  /** ID of RocketLauncher */
  static const int W_ROCKETLAUNCHER = 12;
  /** Number of weapons there are. */
  static const int WEAPON_COUNT = 13;

  
  /** A vector that holds the details of different weapons. Use the
   * weapon-ID's to index this table.
   */
  static std::vector<Specs> weaponlist;
  /** A vector that has the maximum number of ammunition the player can
   * carry.
   */
  static std::vector<int> maxPlayerAmmo;
  /** A vector that has the number of ammunition different ammo-bonuses
   * give.
   */
  static std::vector<int> ammoPacks;
  
  /** The initial velocity of the grenade when thrown. */
  static const eng2d::Vec2D GRENADE_INITIAL_SPEED;
  /** Number of fragments the grenade emit when it explodes. */
  static int GRENADE_FRAGMENTS;
  /** Maximum number of grenades the player can carry. */
  static int MAX_GRENADES;
  /** Number of grenades the player gets from grenadepack-bonus. */
  static int GRENADE_PACK_SIZE;
  
  
  /** Initializes the weapon tables. */
  static void init();
  
  /** Cleans up the weapontables. */
  static void cleanup();
  
  /** Returns the technical details of a weapon.
   * @param     w                 Type of the weapon.
   * @return    A structure that holds the weapon data.
   */
  static Specs& getWeaponSpecs( int w );
  
  /** Returns the maximum number of ammunition the player can carry.
  * @param      w                 Type of the weapon
  * @return     Number of ammos
  */
  static int getMaxAmmo(int w );
  
  /** Returns the number of ammunition the player gets when he picks up a
   * weapon.
   * @param     w                 Type of the weapon.
   * @return    Number of ammunition.
   */
  static int getAmmoPack(int w);
  
  
  
  ///
  /// Constructors, destructor, operators
  /// ===================================
  
  /** Constructs new weapon of given type.
   * @param     id                ID of the weapon.
   */
  Weapon( WeaponID id );
  
  /** A copy constructor.
   * @param     rW                Another Weapon
   */
  Weapon( const Weapon& rW );
  
  /** Destructor.
   */
  ~Weapon();
  
  /** Assignment operator
   * @param     rW                Reference to other Weapon
   * @return    This weapon.
   */
  Weapon& operator = ( const Weapon& rW );
  
  
  ///
  /// Getter methods
  /// ==============
  
  /** Returns the technical details of this weapon.
   * @return    Reference to struct that holds the information of this
   *            weapon type.
   */
  Specs& getSpecs() const;
  
  
  /** Returns the id-code of this weapon.
   * @return    An iD code.
   */
  WeaponID  id() const;
  
  /** Returns the reloading delay this weapon has.
   * @return    Reloading delay.
   */
  int reload() const;
  
  /** Returns the amount of damage the bullets cause.
   * @return    Damagepoints caused by the bullets of this weapon.
   */
  int damage() const;
  
  /** Returns the velocity of the bullets.
   * @return    Velocity of the bullets of this weapon.
   */
  float velocity() const;
  
  /** Returns the range the bullets can fly.
   * @return    Max distance from shooting point.
   */
  float range() const;
  
  /** Returns the radius of the bullets of this weapon.
   * @return    Radius of the bullets
   */
  float radius() const;
  
  /** Returns the deviation of the bullets.
   * @return    The deviation angle. 256 = full circle
   */
  int deviation() const;
  
  /** Tells if the bullets of this weapon should make sound as they hit the
   * tilemap.
   * @return    <code>true</code> if we should make sound
   */
  bool hitSound() const;
  
  /** Tells if we should leave the bullettrails.
   * @return    <code>true</code> if we should leave the bullettrails.
   */
  bool bulletTrails() const;
  
  /** Tells if the bullets of this weapon bounce from the walls.
   * @return    <code>true</code> if the bullets do bounce from the walls.
   */
  bool bouncingBullets() const;
  
  
  /** Returns the ID of the sound that represents this weapon. These values
   * are maintained in SoundPlayer - class.
   * @return    Index of the sample.
   */
  int getSoundID() const;
  
protected:

  ///
  /// Members
  /// =======
  
  /** ID of this weapon. */
  WeaponID  m_weaponID;
};


};  // end of namespace


#endif

/**
 * Version history
 * ===============
 * $Log: weapon.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:30:54+02  lamminsa
 * RocketLauncher weapon added.
 *
 */
 
